Public Interface Control
  Sub Power()
  Sub VolumeUp()
  Sub VolumeDown()
  Sub ChannelUp()
  Sub ChannelDown()
  Sub SelectChannel(ByVal Value As String)
  Property Channel() As String
  Sub Display()
  Event PowerChange(ByVal IsOn As Boolean)
End Interface

Public Class RemoteControl
  Implements Control
  Private FTV As Television

  Public Event PowerChange(ByVal IsOn As Boolean) Implements Control.PowerChange

  Protected Overridable Sub OnPowerChange()
    RaiseEvent PowerChange(FTV.IsOn)
  End Sub

  Public Sub New(ByVal TV As Television)
    MyBase.New()
    FTV = TV
  End Sub

  Public Sub ChannelDown() Implements Control.ChannelDown
    FTV.ChannelDown()
  End Sub

  Public Sub ChannelUp() Implements Control.ChannelUp
    FTV.ChannelUp()
  End Sub

  Public Sub VolumeDown() Implements Control.VolumeDown
    FTV.VolumeDown()
  End Sub

  Public Sub VolumeUp() Implements Control.VolumeUp
    FTV.VolumeUp()
  End Sub

  Public Sub SelectChannel(ByVal Value As String) Implements Control.SelectChannel
    FTV.SelectChannel(Value)
  End Sub

  Public Property Channel() As String Implements Control.Channel
    Get
      Return FTV.Channel
    End Get
    Set(ByVal Value As String)
      FTV.Channel = Value
    End Set
  End Property

  Public Sub Display() Implements Control.Display
    FTV.Display()
  End Sub

  Public Sub Power() Implements Control.Power
    FTV.Power()
    OnPowerChange()
  End Sub
End Class

Public Class Television
  Implements Control

#Region " Public Members "
  Public Event PowerChange(ByVal IsOn As Boolean) Implements Control.PowerChange

  Public ReadOnly Property IsOn() As Boolean
    Get
      Return FIsOn
    End Get
  End Property

  Public Sub New(ByVal Display As AxMediaPlayer.AxMediaPlayer)
    MyBase.New()
    FDisplay = Display
  End Sub

  Public Sub Power() Implements Control.Power
    FIsOn = Not FIsOn
    Changed()
  End Sub

  Public Sub ChannelDown() Implements Control.ChannelDown
    ChangeChannel(-1)
  End Sub

  Public Sub ChannelUp() Implements Control.ChannelUp
    ChangeChannel(1)
  End Sub

  Public Sub VolumeDown() Implements Control.VolumeDown
    ChangeVolume(-50)
  End Sub

  Public Sub VolumeUp() Implements Control.VolumeUp
    ChangeVolume(50)
  End Sub

  Public Sub SelectChannel(ByVal Value As String) Implements Control.SelectChannel
    Static Temp As String
    Temp = Temp & Value
    Display(Temp)

    If (Temp.Length() >= 2) Then
      If (ValidChannel(Temp)) Then
        Channel = Temp
      End If
      Temp = ""
    End If

  End Sub

  Public Overloads Sub Display(ByVal Value As String)
    FDisplay.CreateGraphics.DrawString( _
        Value, FDisplay.Font, Brushes.GreenYellow, _
        10, 10)
  End Sub

  Public Overloads Sub Display() Implements Control.Display
    If (IsOff()) Then Exit Sub
    Display(FChannel)
  End Sub

  Public Property Channel() As String Implements Control.Channel
    Get
      Return FChannel
    End Get

    Set(ByVal Value As String)
      FChannel = Value
      NewChannel()
    End Set
  End Property

#End Region

#Region " Protected Members "
  Protected Overridable Sub OnPowerChange()
    RaiseEvent PowerChange(IsOn)
  End Sub
#End Region

#Region " Private Members "

  Private FDisplay As AxMediaPlayer.AxMediaPlayer
  Private FChannel As String = "02"
  Private FIsOn As Boolean = False

  Private Function IntegerChannel() As Integer
    Return CInt(FChannel)
  End Function

  Private Sub ChangeChannel(ByVal Amount As Integer)
    If (IsOff()) Then Exit Sub

    Dim Temp As Integer = CInt(FChannel) + Amount

    Select Case Temp
      Case Is < 2 : Channel = "09"
      Case Is > 9 : Channel = "02"
      Case Else : Channel = "0" & Temp
    End Select
  End Sub

  Private Sub Changed()
    OnPowerChange()
    If (IsOn) Then
      TurnOn()
    Else
      TurnOff()
    End If
  End Sub

  Private Function IsOff() As Boolean
    Return Not IsOn
  End Function
  Private Sub TurnOff()
    FDisplay.Stop()
    FDisplay.Hide()
  End Sub

  Private Overloads Function GetChannelFile() As String
    Return GetChannelFile(FChannel)
  End Function

  Private Function Exists(ByVal FileName As String) As Boolean
    Return System.IO.File.Exists(FileName)
  End Function

  Private Sub Play(ByVal FileName As String)
    If (Not Exists(FileName)) Then Exit Sub

    Try
      Display()
      FDisplay.FileName = FileName
      FDisplay.Play()
    Catch
    End Try

  End Sub

  Private Sub NewChannel()
    FDisplay.Stop()
    FDisplay.Show()
    Play(GetChannelFile())
  End Sub

  Private Overloads Function GetChannelFile(ByVal Value As String) As String

    Static Channels() As String = _
      {"radar.avi", "radar.avi", "clock.avi", "globe.avi", _
       "radar.avi", "fox.avi", "speedis.avi", _
       "radar.avi", "valve.avi", "radar.avi"}

    Try
      Return "c:\temp\channels\" & Channels(CInt(Value))
    Catch
      Return ""
    End Try
  End Function

  Private Sub TurnOn()
    NewChannel()
  End Sub

  Private Sub ChangeVolume(ByVal Value As Integer)
    If (IsOff()) Then Exit Sub
    Try
      FDisplay.Volume += Value
    Catch
    End Try
  End Sub

  Private Overloads Function ValidChannel(ByVal Value As Integer) As Boolean
    Return (Value > 1) AndAlso (Value < 10)
  End Function

  Private Overloads Function ValidChannel(ByVal Value As String) As Boolean
    Return ValidChannel(CInt(Value))
  End Function
#End Region

End Class